/*
  Name: astmanager.cpp
  Copyright: Under the Mozilla Public License Version 1.1 or later
  Author: Nick Knight
  Date: 20/02/04 15:00
  Description: 
*/
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Asttapi.
 *
 * The Initial Developer of the Original Code is
 * Nick Knight.
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s): none
 *
 * Alternatively, the contents of this file may be used under the terms of
 * the GNU General Public License Version 2 or later (the "GPL")
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

// The debugger can't handle symbols more than 255 characters long.
// STL often creates symbols longer than that.
// When symbols are longer than 255 characters, the warning is issued.
#pragma warning(disable:4786)


#include "astmanager.h" // class's header file
#include "wavetsp.h"
#include ".\astmanager.h"

#include <string>
#include <algorithm>

////////////////////////////////////////////////////////////////////////////////
// Function astManager::astManager()
//
// Class constructor.
//
////////////////////////////////////////////////////////////////////////////////
astManager::astManager(void)
{	
	this->LOGGEDIN = FALSE;
	//to indicate that this is not connected.
	this->s = INVALID_SOCKET;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::~astManager()
//
// Class destructor.
//
////////////////////////////////////////////////////////////////////////////////
astManager::~astManager()
{
	this->dropConnection();
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setHost(std::string astHost)
//
// Sets the hostname of the asterisk server we are connecting to.
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setHost(std::string astHost)
{
    this->host = astHost;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setPort(Long astPort)
//
// Sets the port number for the server we are connecting to.
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setPort(DWORD astPort)
{
    this->port = astPort;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::usernamePassword(std::string username, std::string password)
//
// Sets the hostname of the asterisk server we are connecting to.
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setUsernamePassword(std::string username, std::string password)
{
    this->user = username;
    this->pass = password;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setOriginator(std::string channel)
//
// Sets call originator i.e. "Sip/nick"
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setOriginator(std::string channel)
{
    this->originator = channel;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setOutgoingChannel(std::string channel)
//
// Sets call outgoing channel i.e. "CAPI:<number>:"
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setOutgoingChannel(std::string channel)
{
    this->outGoingChannel = channel;
	this->context = "";
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setInBoundChannel(std::string channel)
//
// Sets call inbound channel channel i.e. "Sip/nick", this can 
// also be a regex if set in the function below
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setInBoundChannel(std::string channel)
{
	this->incomingChannel = channel;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setOutgoingChannel(std::string channel)
//
// Sets call outgoing channel i.e. "CAPI:<number>:"
//
////////////////////////////////////////////////////////////////////////////////
void astManager::useInBoundRegex(bool YesNo)
{
	this->useInComingRegex = YesNo;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setOutgoingChannel(std::string channel)
//
// Sets context i.e. "internal"
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setContext(std::string cont)
{
    this->context = cont;
	this->outGoingChannel = "";
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::setOutgoingChannel(std::string callerid)
//
// Sets callerid i.e. "Nick" <328476284623324>
// This is for the outgoing caller ID (call origination)
//
////////////////////////////////////////////////////////////////////////////////
void astManager::setCallerID(std::string callerid)
{
    this->callerid = callerid;
}

////////////////////////////////////////////////////////////////////////////////
//
// Actions
//
// Now we have set the object up - what can we do?
//
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// Function astManager::connect(void)
//
// Connects to the asterisk server if succsessful sets this->s to the socket
// and returns the socket else returns the error code.
//
////////////////////////////////////////////////////////////////////////////////
SOCKET astManager::astConnect(void)
{
    struct sockaddr_in sa;
    struct hostent     *hp;
    //SOCKET s;
	if ( this->s != INVALID_SOCKET )
	{
		return this->s;
	}
    
    hp = gethostbyname(host.c_str());
    if (hp == NULL) // we don't know who this host is
    {
        this->error = "Unknown host";
        this->s = INVALID_SOCKET;
        return INVALID_SOCKET;
    }

    memset(&sa,0,sizeof(sa));
    memcpy((char *)&sa.sin_addr, hp->h_addr, hp->h_length);   // set address 
    sa.sin_family = hp->h_addrtype;
    sa.sin_port = htons((u_short)port);

    this->s = socket(hp->h_addrtype, SOCK_STREAM, 0);
    if (this->s == INVALID_SOCKET)
    {
        this->error = "Unable to create socket";
        this->s = INVALID_SOCKET;
        return INVALID_SOCKET;
    }

    // try to connect to the specified socket 
    if (connect(this->s, (struct sockaddr *)&sa, sizeof sa) == SOCKET_ERROR) 
    {
        this->error = "Unable to connect";
        closesocket(this->s);
        this->s = INVALID_SOCKET;
        return INVALID_SOCKET;
    }
    return this->s;
}

////////////////////////////////////////////////////////////////////////////////
// Function sendCommand(std::string command)
//
// Private function
//
/* TODO (#1#): the send command function needs to wait for a 
               responce to check it has been accepted by the 
               server */
// Sends the formsatted command to the asterisk server
//
////////////////////////////////////////////////////////////////////////////////
DWORD astManager::sendCommand(std::string command)
{
    send(this->s, command.c_str(), (int)command.length() , 0);
	TSPTRACE("%s",command.c_str());
	return true;
}


////////////////////////////////////////////////////////////////////////////////
// Function astManager::login(std::string user, std::string password)
//
// Private function
//
// Opens the connection to the server and logs in. Returns 1 on sucsess
// if unsucsessful will close the socket down otherwise it will open a socket
// and leave it open.
//
////////////////////////////////////////////////////////////////////////////////
DWORD astManager::login(void)
{
    std::string command;

	if ( this->LOGGEDIN == TRUE )
	{
		return 1;
	}
    
    command = "Action: Login\r\n";
    //send the command
    this->sendCommand(command);
      
    command = "UserName: ";
    command += this->user;
    command += "\r\n";
    //send the command
    this->sendCommand(command);
      
    command = "Secret: ";
    command += this->pass;
    command += "\r\n\r\n";
    //send the command    
    this->sendCommand(command);
    
    return 1;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::originate(std::string destAddress)
//
// Originates a call, when this is called, asterisk will call the originator
// i.e. this->originator, and then call this->outGoingChannel:destAddress
// if that makes sence! Returns 1 on succsess otherwise returns 0.
//
////////////////////////////////////////////////////////////////////////////////
DWORD astManager::originate(std::string destAddress)
{
    //What communication do we need with Asterisk to
    //complete this?
    //
    // 1. Login
    // 2. Originate - we use the application Dial for this
    // 
    // When the object is created, we then login and set up
    // all of the parameters so in this function all we need
    // to do is perform the origination.
    if ( this->s == INVALID_SOCKET ) 
    {
        if ( INVALID_SOCKET == this->astConnect() )
        {
                return 0;
        }
    }

	//if the socket is invalid then we need to login as well.
	if ( 0 == this->login() ) 
	{
		return 0;
	}

	std::string command;

	command = "Action: Originate\r\n";
	//send the command
	this->sendCommand(command);

	command = "Channel: ";
	command += this->originator;
	command += "\r\n";
	//send the command
	this->sendCommand(command);

	if ( this->outGoingChannel != "" )
	{
		command = "Application: Dial\r\n";
		//send the command
		this->sendCommand(command);

		//Ensure that we are sending nothing but numbers to Asterisk
		std::string::iterator it;
		std::string tempStr("");

		for ( it = destAddress.begin(); it != destAddress.end(); it++ )
		{
			if ( *it >= '0' && *it <= '9' )
			{
				tempStr += *it;
			}
		}

		command = "Data: ";
		command += this->outGoingChannel;
		command += tempStr;
		command += "\r\n\r\n";
		//send the command
		this->sendCommand(command);
	}
	else
	{
		//call out by using a context
		//Exten: 1234 
		//Context: default 
		//Callerid: "User" <714-555-1212>

		//Ensure that we are sending nothing but numbers to Asterisk
		std::string::iterator it;
		std::string tempStr("");

		for ( it = destAddress.begin(); it != destAddress.end(); it++ )
		{
			if ( *it >= '0' && *it <= '9' )
			{
				tempStr += *it;
			}
		}
		command = "Exten: ";
		command += tempStr;
		command += "\r\n";
		//send the command
		this->sendCommand(command);

		command = "Priority: 1";
		command += "\r\n";
		this->sendCommand(command);

		//if we set the caller ID?
		if ( this->callerid != "" )
		{
			command = "Callerid: ";
			command += this->callerid;
			command += "\r\n";
			//send the command
			this->sendCommand(command);
		}

		command = "Context: ";
		command += this->context;
		command += "\r\n\r\n";
		//send the command
		this->sendCommand(command);


	}
    
    return 0;
}



////////////////////////////////////////////////////////////////////////////////
// Function astManager::originate(std::string destAddress)
//
// This function will wait for a time for the a messgage from Asterisk
// te end of a message is marked by a \r\n\r\n
////////////////////////////////////////////////////////////////////////////////
astEvent* astManager::waitForMessage(void)
{
	BEGIN_PARAM_TABLE("astManager::waitForMessage")
		DWORD_IN_ENTRY(0)
	END_PARAM_TABLE()

	astEvent *ev = new astEvent;

	DWORD numBytes;
	/* TODO (#1#): need to have a timeout strategy in case of error and better error checking */
	char recvBuf[2];
	std::string rawMessage;
	
	while ( rawMessage.find("\r\n\r\n") == -1 )
	{
		numBytes = recv ( this->s, recvBuf, 1, 0);

		if ( numBytes == 0 || numBytes == SOCKET_ERROR )
		{
			//if we get here the chances are that the socket has
			//been closed down.
			return NULL;
		}

		//An asterisk message is very similar to other protocols using
		//\r\n at the end of an item, and \r\n\r\n at the end of a message
		recvBuf[1] = 0;	//ensure NULL termination
		rawMessage += recvBuf;
		//TSPTRACE("%s",rawMessage.c_str());

	}
	TSPTRACE("Found terminator\r\n");

	//When we get here we have a complete message which we need to split up 
	//into a usable format (I suppose we could also keep as is - but I prefer it this way!)
	size_t carPos;
	while ( ( carPos = rawMessage.find("\r\n") ) != -1)
	{
		std::string line = rawMessage.substr(0,carPos);
		rawMessage = rawMessage.substr(carPos+2, rawMessage.length());
		//When we are down to the line level it is in the format
		//header: data
		size_t dataPos;
		if ( ( dataPos = line.find(":")) != -1 )
		{
			std::string header = line.substr(0,dataPos);
			std::string data = line.substr(dataPos+2, line.length());
			std::transform(header.begin(), header.end(), header.begin(), tolower);
			
			// a little logging info
			TSPTRACE("header: %s, data: %s\r\n",header.c_str(), data.c_str());

			if ( header == "event" )
			{
				if ( data == "Shutdown" )
				{
					ev->event = evShutdown;
				}
				else if ( data == "Hangup" )
				{
					ev->event = evHangup;
				}
				else if ( data == "Rename " )
				{
					ev->event = evRename;
				}
				else if ( data == "Hangup" )
				{
					ev->event = evHangup;
				}
				else if ( data == "Newcallerid" )
				{
					ev->event = evNewcallerid;
				}
				else if ( data == "Newchannel" )
				{
					ev->event = evNewchannel;
				}
				else if ( data == "Link" )
				{
					ev->event = evLink;
				}
				else if ( data == "Unlink" )
				{
					ev->event = evUnlink;
				}
				else if ( data == "ExtensionStatus" )
				{
					ev->event = evExtensionStatus;
				}
				else if ( data == "Reload" )
				{
					ev->event = evReload;
				}
				else if ( data == "Newexten" )
				{
					ev->event = evNewexten;
				}
				else if ( data == "Newstate" )
				{
					ev->event = evNewstate;
				}
			}
			else if ( header == "Uniqueid" )
			{
				ev->uniqueid = data;
			}
			else if ( header == "uniqueid1" )
			{
				ev->uniqueid1 = data;
			}
			else if ( header == "uniqueid2" )
			{
				ev->uniqueid2 = data;
			}
			else if ( header == "callerid" )
			{
				ev->callerId = data;
			}
			else if ( header == "extension" )
			{
				ev->extension = data;
			}
			else if ( header == "context" )
			{
				ev->context = data;
			}
			else if ( header == "state" )
			{
				if ( data == "Ringing" )
				{
					ev->state = stRinging;
				}
				else if ( data == "Ring" )
				{
					ev->state = stRing;
				}
				else if ( data == "Dialing" )
				{
					ev->state = stDial;
				}
				else if ( data == "Down" )
				{
					ev->state = stDown;
				}
				else if (data == "Up" )
				{
					ev->state = stUp;
				}
			}
			else if ( header == "channel" )
			{
				ev->channel = data;
			}
		}

	}

	return ev;
}

////////////////////////////////////////////////////////////////////////////////
// Function astManager::originate(std::string destAddress)
//
// Call this function to drop whatever call has been originated by this class. 
// This is still to be completed.
void astManager::dropChannel(std::string channel)
{
	std::string command;

	command = "Action: Hangup\r\n";
	//send the command
	this->sendCommand(command);

	command = "Channel: ";
	command += channel;
	command += "\r\n\r\n";

	this->sendCommand(command);
}
////////////////////////////////////////////////////////////////////////////////




DWORD astManager::dropConnection(void)
{
	if (this->s != INVALID_SOCKET)
	{
		closesocket(this->s);
	}

	this->s = INVALID_SOCKET;

	return 0;
}
